from db import get_db_connection
from utils import check_password, hash_password
from jwt_auth import generate_jwt
import re

def is_valid_input(value):
    """Validasi input agar tidak mengandung karakter mencurigakan, tetapi memperbolehkan spasi."""
    return bool(re.match(r"^[a-zA-Z0-9_.\-\s]+$", value))

def login_admin(nama, password):
    """Login Admin dengan JWT dan validasi input."""
    if not is_valid_input(nama):
        return {"error": "Format username tidak valid!"}

    db = get_db_connection()
    if not db:
        return {"error": "Database tidak dapat dihubungi"}

    cursor = db.cursor()
    cursor.execute("SELECT id, nama, password, level FROM admin WHERE nama = %s", (nama,))
    result = cursor.fetchone()
    db.close()

    if result and check_password(password, result[2]):
        token = generate_jwt(result[0], result[1], result[3])
        return {"id": result[0], "nama": result[1], "level": result[3], "token": token}

    return {"error": "Login gagal! Periksa username/password."}


def login_user(nik, password):
    """Login Nasabah dengan JWT."""
    if not is_valid_input(nik):
        return {"error": "Format NIK tidak valid!"}

    db = get_db_connection()
    cursor = db.cursor()
    
    cursor.execute("SELECT nik, nama, password FROM nasabah WHERE nik = %s", (nik,))
    result = cursor.fetchone()
    db.close()

    if result and check_password(password, result[2]):
        token = generate_jwt(result[0], result[1], "nasabah")
        return {"id": result[0], "nama": result[1], "level": "nasabah", "token": token}

    return {"error": "Login gagal! Periksa NIK/password."}

def is_valid_email(email):
    """Validasi email dengan regex."""
    return bool(re.match(r"^[\w\.-]+@[\w\.-]+\.\w+$", email))

def register_user(nik, nama, nohp, alamat, email, password, foto_ktp, cabang):
    """Registrasi Nasabah dengan validasi input."""
    if not is_valid_email(email):
        return {"error": "Format email tidak valid!"}
    
    if not is_valid_input(nik):
        return {"error": "Format NIK tidak valid!"}
    
    if not is_valid_input(nama):
        return {"error": "Format nama tidak valid!"}
    
    if not is_valid_input(alamat):
        return {"error": "Format alamat tidak valid!"}
    
    if not is_valid_input(cabang):
        return {"error": "Format cabang tidak valid!"}
    
    if not is_valid_input(nohp):
        return {"error": "Format nomor HP tidak valid!"}
    
    db = get_db_connection()
    if not db:
        return {"error": "Database tidak dapat dihubungi"}

    cursor = db.cursor()
    cursor.execute("SELECT nik FROM nasabah WHERE nik = %s", (nik,))
    if cursor.fetchone():
        return {"error": "NIK sudah terdaftar!"}

    hashed_password = hash_password(password)
    cursor.execute(
        "INSERT INTO nasabah (nik, nama, no_hp, alamat, email, password, foto_ktp, cabang) VALUES (%s, %s, %s, %s, %s, %s, %s, %s)",
        (nik, nama, nohp, alamat, email, hashed_password, foto_ktp, cabang)
    )
    db.commit()
    db.close()

    return {"message": "Registrasi berhasil!"}

def register_cabang(nama, alamat, nohp, email, password, logo):
    """Registrasi Admin Cabang."""

    if not is_valid_email(email):
        return {"error": "Format email tidak valid!"}
    
    if not is_valid_input(nama):
        return {"error": "Format nama tidak valid!"}
    
    if not is_valid_input(alamat):
        return {"error": "Format alamat tidak valid!"}
    
    if not is_valid_input(nohp):
        return {"error": "Format nomor HP tidak valid!"}
    
    db = get_db_connection()
    cursor = db.cursor()

    try:
        # Periksa apakah nama atau email sudah digunakan
        cursor.execute("SELECT nama FROM admin WHERE nama = %s OR email = %s", (nama, email))
        if cursor.fetchone():
            return {"error": "Nama atau email sudah terdaftar!"}

        # Hash password sebelum disimpan
        hashed_password = hash_password(password)

        # Simpan data ke database
        cursor.execute(
            "INSERT INTO admin (nama, alamat, no_hp, email, password, level, logo) VALUES (%s, %s, %s, %s, %s, %s, %s)",
            (nama, alamat, nohp, email, hashed_password, "cabang", logo),
        )
        db.commit()

        # Pastikan data benar-benar tersimpan
        if cursor.lastrowid:
            return {"message": "Registrasi berhasil!"}
        else:
            return {"error": "Gagal menyimpan data."}

    except Exception as e:
        db.rollback()  # Kembalikan perubahan jika terjadi error
        return {"error": f"Terjadi kesalahan: {str(e)}"}

    finally:
        cursor.close()
        db.close()